/* --------------------------------- player.c ------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Manage participating players
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fly.h"

static PLAYER	*players = 0;		/* active list */

extern PLAYER * FAR
player_add (PACKET *pack)
{
	PLAYER	*pl;

	if (!NEW (pl))
		return (0);

	pl->next = players;
	players = pl;
	pl->netport = pack->netport;
	pl->timeout = st.PlayerTimeout;
	memcpy (pl->address, pack->address, LADDRESS);
	PlName (pl);
	return (pl);
}

extern PLAYER * FAR
player_delete (PLAYER *player)
{
	PLAYER	*pl, *pp;

	if (!player)
		return (0);

	for (pp = 0, pl = players; pl; pp = pl, pl = pl->next) {
		if (pl == player) {
			pl = pl->next;
			if (pp)
				pp->next = pl;
			else
				players = pl;
			break;
		}
	}

	DEL0 (player);

	return (pl);
}

extern void FAR
players_delete (void)
{
	PLAYER	*pl;

	for (pl = players; pl;)
		pl = DEL (pl);
	players = 0;
}

extern void FAR
player_remove (PLAYER *player)
/*
 * Remove all objects of a remote player from our world.
*/
{
	OBJECT	*p;

	for (p = CO; p; p = p->next) {
		if ((p->flags & F_IMPORTED) && p->rplayer == player)
			p->flags |= F_DEL|F_MOD;
	}
	if (player->flags & RMT_PLAYING) {
		player->flags &= ~RMT_PLAYING;
		player->flags |= RMT_ACTIVE;
		netport_count (player, -1);
	}
}

extern void FAR
players_remove (void)
/*
 * Remove all imported objects from our world.
*/
{
	PLAYER	*pl;

	for (pl = players; pl; pl = pl->next)
		player_remove (pl);
}

static void FAR
player_flush (PLAYER *pl)
/*
 * Delete all noise-messages from a player.
*/
{
	PACKET	*pack;

	for (pack = pl->incoming; pack;) {
		pack = packet_del (pack);
		if (pl->flags & RMT_PLAYING)
			++st.stats[4];
		else
			++st.stats[55];
	}
	pl->incoming =  0;
	pl->tail = 0;
}

extern void FAR
players_flush (void)
/*
 * Delete all noise-messages.
*/
{
	PLAYER	*pl;

	for (pl = players; pl; pl = pl->next)
		player_flush (pl);
}

extern void FAR
players_purge (void)
/*
 * purge silent players (life was never fair).
*/
{
	PLAYER	*pl;

	for (pl = players; pl; pl = pl->next) {
		if (!(pl->flags & (RMT_NOTIDLE & ~RMT_ACTIVE)))
			continue;
		if (pl->timeout < st.present) {
			LogPrintf ("%s ", Tm->Ctime ());
			MsgWPrintf (-100, "Timed: %s", pl->name);
			if (pl->flags & RMT_PLAYING)
				remote_noplay (pl, 0);
			player_flush (pl);
			player_remove (pl);
			pl->flags &= ~RMT_NOTIDLE;
		}
	}
}

extern PLAYER * FAR
player_find (PACKET *pack)
/*
 * Find a player by address.
*/
{
	PLAYER	*pl;

	for (pl = players; pl; pl = pl->next)
		if (pl->netport == pack->netport &&
		    !memcmp (pl->address, pack->address, LADDRESS))
			break;
	return (pl);
}

extern PLAYER * FAR
player_active (PACKET *pack)
/*
 * Make sure a player is registered.
*/
{
	PLAYER	*pl;

	if (!(pl = player_find (pack)))
		pl = player_add (pack);
	if (pl && !(pl->flags & RMT_NOTIDLE))
		pl->flags |= RMT_ACTIVE;
	return (pl);
}

extern PLAYER * FAR
player_next (PLAYER *pl)
/*
 * Provide access to players list.
*/
{
	if (pl)
		return (pl->next);
	else
		return (players);
}

extern int FAR
players_init (void)
{
	players = 0;
	return (0);
}

extern void FAR
players_term (void)
{
	players_remove ();
	players_delete ();
}
